/**
 * \file caam_op_key_import.c
 *
 * \brief Architecture specific implementation for keystore key import
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2017 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include <stdlib.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <string.h>
#include <sdc/arch/errors.h>
#include <sdc/arch/common.h>
#include <sdc/arch/perm.h>
#include <private/sdc_arch.h>
#include <sdc/daemon/arch/sdc_daemon_arch.h>
#include "caam.h"
#include <linux/caam_ee.h>
#include <linux/caam_km.h>

sdc_error_t sdc_arch_wrapped_storage_key_to_daemon_data(
    sdc_session_t *session,
    const sdc_keystore_key_cfg_t *key_config,
    const sdc_wrapped_key_t *wrapped_key,
    uint8_t **daemon_data, size_t *daemon_data_len)
{
    sdc_error_t err = SDC_OK;
    struct caam_ee_prep_wrapped_key_import iodata;
    int res;
    int res_errno;
    size_t required_daemon_data_len;

    (void)key_config;

    /*
     * Output data will be passed to khd in order to import the wrapped key.
     * The output data starts with some info (encap_info) on limits for
     * permissions/authenticity/confidentiality.
     * Followed by the IMX6 specific encapsulation of the previously wrapped key.
     * Only kernel will be able to decapsulate
     *
     * Note: key_secret_len == wrapped_key_len
     */
    required_daemon_data_len = CAAM_KEY_ENCAP_INFO_LEN + CAAM_KM_BLOB_ADD +
            wrapped_key->wrapped_len;

    *daemon_data_len = 0;

    *daemon_data = malloc(required_daemon_data_len);
    if (*daemon_data == NULL)
        return SDC_NO_MEM;

    if ((wrapped_key->wrap_type->alg != SDC_WRAP_ALG_AES) ||
            (wrapped_key->wrap_type->blk != SDC_WRAP_BLK_CCM))
        /* imx6 only supports AES-CCM */
        err = SDC_OP_NOT_SUPPORTED;

    iodata.flags = 0;
    iodata.wrapped_key = (uint8_t*)wrapped_key->wrapped_data;
    iodata.wrapped_key_len = (size_t)wrapped_key->wrapped_len;
    iodata.wrapped_iv = (uint8_t*)wrapped_key->iv;
    iodata.wrapped_iv_len = (size_t)wrapped_key->iv_len;
    iodata.wrapped_tag = (uint8_t*)wrapped_key->tag;
    iodata.wrapped_tag_len = (size_t)wrapped_key->tag_len;
    iodata.key_import_data = (uint8_t*)(*daemon_data);
    iodata.key_import_data_len = (size_t)(required_daemon_data_len);

    if (err == SDC_OK) {
        res = ioctl(session->arch.fd, CAAM_EE_PREPARE_WRAPPED_KEY_IMPORT, &iodata);

        if (res != 0) {
            res_errno = errno;
            err = error_from_ioctl_errno(res_errno, CAAM_EE_PREPARE_WRAPPED_KEY_IMPORT, iodata.flags);
        } else {
            *daemon_data_len = iodata.key_import_data_len; /* store used len */
        }
    }

    if (*daemon_data_len == 0) {
        free(*daemon_data);
        *daemon_data = NULL;
    }

    return err;
}

